import sql from "@/app/api/utils/sql";

// Get user settings
export async function GET(request) {
  try {
    const { searchParams } = new URL(request.url);
    const userId = searchParams.get("userId") || "demo";

    let result = await sql("SELECT * FROM user_settings WHERE user_id = $1", [
      userId,
    ]);

    // Create default settings if they don't exist
    if (result.length === 0) {
      result = await sql(
        `INSERT INTO user_settings (user_id, is_premium, notification_enabled, reminder_time)
         VALUES ($1, false, true, '20:00')
         RETURNING *`,
        [userId],
      );
    }

    return Response.json({ success: true, settings: result[0] });
  } catch (error) {
    console.error("Error fetching user settings:", error);
    return Response.json(
      { success: false, error: error.message },
      { status: 500 },
    );
  }
}

// Update user settings
export async function PUT(request) {
  try {
    const body = await request.json();
    const {
      userId = "demo",
      isPremium,
      notificationEnabled,
      reminderTime,
    } = body;

    const updates = [];
    const values = [];
    let paramIndex = 1;

    if (isPremium !== undefined) {
      updates.push(`is_premium = $${paramIndex}`);
      values.push(isPremium);
      paramIndex++;
    }
    if (notificationEnabled !== undefined) {
      updates.push(`notification_enabled = $${paramIndex}`);
      values.push(notificationEnabled);
      paramIndex++;
    }
    if (reminderTime !== undefined) {
      updates.push(`reminder_time = $${paramIndex}`);
      values.push(reminderTime);
      paramIndex++;
    }

    updates.push(`updated_at = NOW()`);
    values.push(userId);

    const query = `
      INSERT INTO user_settings (user_id, is_premium, notification_enabled, reminder_time)
      VALUES ($${paramIndex}, $${paramIndex + 1}, $${paramIndex + 2}, $${paramIndex + 3})
      ON CONFLICT (user_id) DO UPDATE SET ${updates.join(", ")}
      RETURNING *
    `;

    const settingsValues = [
      userId,
      isPremium ?? false,
      notificationEnabled ?? true,
      reminderTime ?? "20:00",
      ...values,
    ];

    const result = await sql(query, settingsValues);

    return Response.json({ success: true, settings: result[0] });
  } catch (error) {
    console.error("Error updating user settings:", error);
    return Response.json(
      { success: false, error: error.message },
      { status: 500 },
    );
  }
}
