import sql from "@/app/api/utils/sql";

// Get all mood entries for a user
export async function GET(request) {
  try {
    const { searchParams } = new URL(request.url);
    const userId = searchParams.get("userId") || "demo";
    const limit = parseInt(searchParams.get("limit") || "100");
    const startDate = searchParams.get("startDate");
    const endDate = searchParams.get("endDate");

    let query = "SELECT * FROM mood_entries WHERE user_id = $1";
    const params = [userId];
    let paramIndex = 2;

    if (startDate) {
      query += ` AND created_at >= $${paramIndex}`;
      params.push(startDate);
      paramIndex++;
    }

    if (endDate) {
      query += ` AND created_at <= $${paramIndex}`;
      params.push(endDate);
      paramIndex++;
    }

    query += ` ORDER BY created_at DESC LIMIT $${paramIndex}`;
    params.push(limit);

    const entries = await sql(query, params);

    return Response.json({ success: true, entries });
  } catch (error) {
    console.error("Error fetching mood entries:", error);
    return Response.json(
      { success: false, error: error.message },
      { status: 500 },
    );
  }
}

// Create a new mood entry
export async function POST(request) {
  try {
    const body = await request.json();
    const { userId = "demo", moodEmoji, moodLabel, note, journalEntry } = body;

    if (!moodEmoji || !moodLabel) {
      return Response.json(
        { success: false, error: "Mood emoji and label are required" },
        { status: 400 },
      );
    }

    const result = await sql(
      `INSERT INTO mood_entries (user_id, mood_emoji, mood_label, note, journal_entry, created_at, updated_at)
       VALUES ($1, $2, $3, $4, $5, NOW(), NOW())
       RETURNING *`,
      [userId, moodEmoji, moodLabel, note || null, journalEntry || null],
    );

    return Response.json({ success: true, entry: result[0] });
  } catch (error) {
    console.error("Error creating mood entry:", error);
    return Response.json(
      { success: false, error: error.message },
      { status: 500 },
    );
  }
}
